/*
Gross growth and components of gross growth, ag and bg carbon density change
from query of FIADB 

Estimates are ratio estimates modified from code provided with evalidator queries
  and based on the growth, removals, and mortality accouning approach within the FIADB
The code generates two tables accoding to stand classifications -- by typegroup versus softwood/hardwood/woodland
The resulting .csv table (e.g., grossGrowth_wComponents_02Nov2022.csv)
  returned from oracle is the union tables and is exactly as it appears in the model

Column names are explained in the model, but briefly:
- columns 1-4 classify the forest stand
- GROSSGROWCAG_TPAPY: gross growth aboveground carbon, tons C/acre/year
- GROSSGROWCBG_TPAPY: gross growth belowground carbon, tons C/acre/year
- FOREST_ACRES: forest area used in the denominator of the estimates, acres
followed by 20 components, which sum to the above totals
  note:
   - T1_ and T2_ are time1 and time2, or the beginning and end of the remeasurement interval
   - _TPAPY are the units for each,  tons C/acre/year 
   - AG_ and BG_ are aboveground and belowground
4 03 groups 
    
See the FIADB database documentaion for additional informaion on data and methods  
https://www.fia.fs.usda.gov/library/database-documentation/index.php#FIADB 

05Oct2023, from 09Dec2022 jes 
*/
SELECT 
  num.conusRegion, 
  num.possibleTypes, 
  num.standOrigin, 
  num.ageBin, 
  num.ESTIMATEag / den.ESTIMATE as grossGrowCag_tpapy, 
  /* RATIO_ESTIMATED_VALUE,  */
  num.ESTIMATEbg / den.ESTIMATE as grossGrowCbg_tpapy, 
  den.ESTIMATE as forest_acres, 
  num.t2_survivor_agdw_ton / den.ESTIMATE as t2_survivorCag_tpapy, 
  num.t2_ingrowth_agdw_ton / den.ESTIMATE as t2_ingrowthCag_tpapy, 
  num.t2_reversion_agdw_ton / den.ESTIMATE as t2_reversionCag_tpapy, 
  num.t2_cut_agdw_ton / den.ESTIMATE as t2_cutCag_tpapy, 
  num.t2_diversion_agdw_ton / den.ESTIMATE as t2_diversionCag_tpapy, 
  num.t2_mortality_agdw_ton / den.ESTIMATE as t2_mortalityCag_tpapy, 
  num.t1_survivor_agdw_ton / den.ESTIMATE as t1_survivorCag_tpapy, 
  num.t1_cut_agdw_ton / den.ESTIMATE as t1_cutCag_tpapy, 
  num.t1_diversion_agdw_ton / den.ESTIMATE as t1_diversionCag_tpapy, 
  num.t1_mortality_agdw_ton / den.ESTIMATE as t1_mortalityCag_tpapy, 
  num.t2_survivor_bgdw_ton / den.ESTIMATE as t2_survivorCbg_tpapy, 
  num.t2_ingrowth_bgdw_ton / den.ESTIMATE as t2_ingrowthCbg_tpapy, 
  num.t2_reversion_bgdw_ton / den.ESTIMATE as t2_reversionCbg_tpapy, 
  num.t2_cut_bgdw_ton / den.ESTIMATE as t2_cutCbg_tpapy, 
  num.t2_diversion_bgdw_ton / den.ESTIMATE as t2_diversionCbg_tpapy, 
  num.t2_mortality_bgdw_ton / den.ESTIMATE as t2_mortalityCbg_tpapy, 
  num.t1_survivor_bgdw_ton / den.ESTIMATE as t1_survivorCbg_tpapy, 
  num.t1_cut_bgdw_ton / den.ESTIMATE as t1_cutCbg_tpapy, 
  num.t1_diversion_bgdw_ton / den.ESTIMATE as t1_diversionCbg_tpapy, 
  num.t1_mortality_bgdw_ton / den.ESTIMATE as t1_mortalityCbg_tpapy 
FROM 
  (
    SELECT 
      conusRegion, 
      coalesce(possibleTypes, 'Unknown') as possibleTypes, 
      coalesce(standOrigin, 'Unknown') as standOrigin, 
      coalesce(ageBin, 'Unknown') as ageBin, 
      sum(ESTIMATED_VALUEag * EXPNS) ESTIMATEag, 
      sum(ESTIMATED_VALUEbg * EXPNS) ESTIMATEbg, 
      sum(t2_survivor_agdw_ton * EXPNS) t2_survivor_agdw_ton, 
      sum(t2_ingrowth_agdw_ton * EXPNS) t2_ingrowth_agdw_ton, 
      sum(t2_reversion_agdw_ton * EXPNS) t2_reversion_agdw_ton, 
      sum(t2_cut_agdw_ton * EXPNS) t2_cut_agdw_ton, 
      sum(t2_diversion_agdw_ton * EXPNS) t2_diversion_agdw_ton, 
      sum(t2_mortality_agdw_ton * EXPNS) t2_mortality_agdw_ton, 
      sum(t1_survivor_agdw_ton * EXPNS) t1_survivor_agdw_ton, 
      sum(t1_cut_agdw_ton * EXPNS) t1_cut_agdw_ton, 
      sum(t1_diversion_agdw_ton * EXPNS) t1_diversion_agdw_ton, 
      sum(t1_mortality_agdw_ton * EXPNS) t1_mortality_agdw_ton, 
      sum(t2_survivor_bgdw_ton * EXPNS) t2_survivor_bgdw_ton, 
      sum(t2_ingrowth_bgdw_ton * EXPNS) t2_ingrowth_bgdw_ton, 
      sum(t2_reversion_bgdw_ton * EXPNS) t2_reversion_bgdw_ton, 
      sum(t2_cut_bgdw_ton * EXPNS) t2_cut_bgdw_ton, 
      sum(t2_diversion_bgdw_ton * EXPNS) t2_diversion_bgdw_ton, 
      sum(t2_mortality_bgdw_ton * EXPNS) t2_mortality_bgdw_ton, 
      sum(t1_survivor_bgdw_ton * EXPNS) t1_survivor_bgdw_ton, 
      sum(t1_cut_bgdw_ton * EXPNS) t1_cut_bgdw_ton, 
      sum(t1_diversion_bgdw_ton * EXPNS) t1_diversion_bgdw_ton, 
      sum(t1_mortality_bgdw_ton * EXPNS) t1_mortality_bgdw_ton 
    FROM 
      (
        SELECT 
          case when (
            cond.stdage between 0 
            and 20
          ) then '0-20 years' when (
            cond.stdage between 21 
            and 40
          ) then '21-40 years' when (
            cond.stdage between 41 
            and 60
          ) then '41-60 years' when (
            cond.stdage between 61 
            and 80
          ) then '61-80 years' when (
            cond.stdage between 81 
            and 100
          ) then '81-100 years' when (
            cond.stdage between 101 
            and 998
          ) then '100+ years' else 'w/o age - row not selectable' end as agebin, 
          refftgrp.meaning as possibleTypes, 
          case when cond.stdorgcd = 1 then 'Planted' else 'Natural' end as standOrigin, 
          case when cond.statecd in(
            9, 10, 11, 23, 24, 25, 33, 34, 36, 42, 44, 
            50, 54
          ) then 'Northeast' when cond.statecd in(17, 18, 19, 29, 39) then 'Central States' when cond.statecd in(20, 31, 38, 46) then 'Great Plains' when cond.statecd in(26, 27, 55) then 'Northern Lake States' when (
            cond.statecd in(40, 48) 
            and cond.unitcd in(3, 4, 5, 6, 7)
          ) then 'Great Plains' when cond.statecd in(1, 5, 21, 22, 28, 40, 47, 48) then 'South Central' when cond.statecd in(12, 13, 37, 45, 51) then 'Southeast' when cond.statecd in(16, 30) then 'Rocky Mountain North' when cond.statecd in(4, 8, 32, 35, 49, 56) then 'Rocky Mountain South' when cond.statecd in(6) then 'Pacific Southwest' when (
            cond.statecd in(41, 53) 
            and cond.unitcd in(0, 1, 2, 5, 6, 7)
          ) then 'Pacific Northwest Westside' when (
            cond.statecd in(41, 53) 
            and cond.unitcd in(3, 4, 8, 9)
          ) then 'Pacific Northwest Eastside' else 'not located' end as conusRegion, 
          pop_stratum.expns EXPNS, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * (
              CASE WHEN BE.ONEORTWO = 2 THEN (
                CASE WHEN (
                  GRM.COMPONENT = 'SURVIVOR' 
                  OR GRM.COMPONENT = 'INGROWTH' 
                  OR GRM.COMPONENT LIKE 'REVERSION%'
                ) THEN (
                  TREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
                ) WHEN (
                  GRM.COMPONENT LIKE 'CUT%' 
                  OR GRM.COMPONENT LIKE 'DIVERSION%' 
                  OR GRM.COMPONENT LIKE 'MORTALITY%'
                ) THEN (
                  TRE_MIDPT.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
                ) ELSE (0) END
              ) ELSE (
                CASE WHEN (
                  GRM.COMPONENT = 'SURVIVOR' 
                  OR GRM.COMPONENT = 'CUT1' 
                  OR GRM.COMPONENT = 'DIVERSION1' 
                  OR GRM.COMPONENT = 'MORTALITY1'
                ) THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
                  TRE_BEGIN.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
                ) ELSE -(
                  PTREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
                ) END ELSE (0) END
              ) END
            )
          ) AS ESTIMATED_VALUEag, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * (
              CASE WHEN BE.ONEORTWO = 2 THEN (
                CASE WHEN (
                  GRM.COMPONENT = 'SURVIVOR' 
                  OR GRM.COMPONENT = 'INGROWTH' 
                  OR GRM.COMPONENT LIKE 'REVERSION%'
                ) THEN (
                  TREE.DRYBIO_bG / 2 / 2000 / PLOT.REMPER
                ) WHEN (
                  GRM.COMPONENT LIKE 'CUT%' 
                  OR GRM.COMPONENT LIKE 'DIVERSION%' 
                  OR GRM.COMPONENT LIKE 'MORTALITY%'
                ) THEN (
                  TRE_MIDPT.DRYBIO_bG / 2 / 2000 / PLOT.REMPER
                ) ELSE (0) END
              ) ELSE (
                CASE WHEN (
                  GRM.COMPONENT = 'SURVIVOR' 
                  OR GRM.COMPONENT = 'CUT1' 
                  OR GRM.COMPONENT = 'DIVERSION1' 
                  OR GRM.COMPONENT = 'MORTALITY1'
                ) THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
                  TRE_BEGIN.DRYBIO_bG / 2 / 2000 / PLOT.REMPER
                ) ELSE -(
                  PTREE.DRYBIO_bG / 2 / 2000 / PLOT.REMPER
                ) END ELSE (0) END
              ) END
            )
          ) AS ESTIMATED_VALUEbg, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT = 'SURVIVOR' THEN TREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_survivor_agdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT = 'INGROWTH' THEN TREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_ingrowth_agdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'REVERSION%' THEN TREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_reversion_agdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'CUT%' THEN TRE_MIDPT.DRYBIO_AG / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_cut_agdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'DIVERSION%' THEN TRE_MIDPT.DRYBIO_AG / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_diversion_agdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'MORTALITY%' THEN TRE_MIDPT.DRYBIO_AG / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_mortality_agdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'SURVIVOR' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_survivor_agdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'CUT1' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_cut_agdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'DIVERSION1' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_diversion_agdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'MORTALITY1' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_mortality_agdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT = 'SURVIVOR' THEN TREE.DRYBIO_bg / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_survivor_bgdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT = 'INGROWTH' THEN TREE.DRYBIO_bg / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_ingrowth_bgdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'REVERSION%' THEN TREE.DRYBIO_bg / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_reversion_bgdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'CUT%' THEN TRE_MIDPT.DRYBIO_bg / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_cut_bgdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'DIVERSION%' THEN TRE_MIDPT.DRYBIO_bg / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_diversion_bgdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'MORTALITY%' THEN TRE_MIDPT.DRYBIO_bg / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_mortality_bgdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'SURVIVOR' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_survivor_bgdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'CUT1' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_cut_bgdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'DIVERSION1' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_diversion_bgdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'MORTALITY1' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_mortality_bgdw_ton 
        FROM 
          FS_FIADB.BEGINEND BE, 
          FS_FIADB.POP_STRATUM POP_STRATUM 
          JOIN FS_FIADB.POP_PLOT_STRATUM_ASSGN POP_PLOT_STRATUM_ASSGN ON (
            POP_STRATUM.CN = POP_PLOT_STRATUM_ASSGN.STRATUM_CN
          ) 
          JOIN FS_FIADB.PLOT PLOT ON (
            POP_PLOT_STRATUM_ASSGN.PLT_CN = PLOT.CN
          ) 
          JOIN FS_FIADB.PLOTGEOM PLOTGEOM ON (PLOT.CN = PLOTGEOM.CN) 
          JOIN FS_FIADB.PLOT PPLOT ON (PLOT.PREV_PLT_CN = PPLOT.CN) 
          JOIN FS_FIADB.COND PCOND ON (PLOT.PREV_PLT_CN = PCOND.PLT_CN) 
          JOIN FS_FIADB.COND COND ON (PLOT.CN = COND.PLT_CN) 
          JOIN FS_FIADB.REF_FOREST_TYPE refft ON (COND.FORTYPCD = refft.VALUE) 
          JOIN FS_FIADB.REF_FOREST_TYPE_GROUP refftgrp ON (refftgrp.value = refft.typgrpcd) 
          JOIN FS_FIADB.TREE TREE ON (
            TREE.CONDID = COND.CONDID 
            AND TREE.PLT_CN = PLOT.CN 
            AND TREE.PREVCOND = PCOND.CONDID
          ) 
          LEFT OUTER JOIN FS_FIADB.TREE PTREE ON (TREE.PREV_TRE_CN = PTREE.CN) 
          LEFT OUTER JOIN FS_FIADB.TREE_GRM_BEGIN TRE_BEGIN ON (TREE.CN = TRE_BEGIN.TRE_CN) 
          LEFT OUTER JOIN FS_FIADB.TREE_GRM_MIDPT TRE_MIDPT ON (TREE.CN = TRE_MIDPT.TRE_CN) 
          LEFT OUTER JOIN (
            SELECT 
              TRE_CN, 
              DIA_BEGIN, 
              DIA_MIDPT, 
              DIA_END, 
              MICR_COMPONENT_AL_FOREST AS COMPONENT, 
              MICR_SUBPTYP_GRM_AL_FOREST AS SUBPTYP_GRM, 
              MICR_TPAGROW_UNADJ_AL_FOREST AS TPAGROW_UNADJ 
            FROM 
              FS_FIADB.TREE_GRM_COMPONENT
          ) GRM ON (TREE.CN = GRM.TRE_CN) 
        WHERE 
          TREE.PREVCOND = PCOND.CONDID 
          AND PCOND.COND_STATUS_CD = 1 
          AND COND.COND_STATUS_CD = 1 
          AND 1 = 1 
         /* conus46 and the 2 split states */

         AND (
            ( pop_stratum.evalid in (12203, 41903, 52103, 61903, 81903, 92103, 
             102103, 121903, 132103, 161903, 172103, 182103, 192103, 202003, 211903, 221903, 
             232103, 242103, 252003, 262103, 272103, 282103, 292103, 301903, 312103, 321903,
             332103, 342003, 351903, 362003, 372203, 382103, 392103, 411903, 422103, 442103, 
             452103, 462103, 471903, 491903, 502103, 512103, 531903, 542103, 552103, 561803	))  -- the 46  
             or
            (POP_PLOT_STRATUM_ASSGN.unitcd in (3,4,5,6,7) and pop_stratum.evalid = 481903)  -- TX west  
            or
            (POP_PLOT_STRATUM_ASSGN.unitcd in (1,2) and pop_stratum.evalid = 482123)  -- TX east 
            or 
            (POP_PLOT_STRATUM_ASSGN.unitcd in (3,4,5,6,7) and pop_stratum.evalid = 402003)  -- OK west   
            or
            (POP_PLOT_STRATUM_ASSGN.unitcd in (1,2) and pop_stratum.evalid = 402003)  -- OK east  
          )
			
        GROUP BY 
          pop_stratum.estn_unit_cn, 
          pop_stratum.cn, 
          plot.cn, 
          plot.lat, 
          plot.lon, 
          pop_stratum.expns, 
          FS_FIADB.EVALIDATORFUNCTIONS.fortypgrpcdLabel(cond.fortypcd), 
          case when (
            cond.stdage between 0 
            and 20
          ) then '0-20 years' when (
            cond.stdage between 21 
            and 40
          ) then '21-40 years' when (
            cond.stdage between 41 
            and 60
          ) then '41-60 years' when (
            cond.stdage between 61 
            and 80
          ) then '61-80 years' when (
            cond.stdage between 81 
            and 100
          ) then '81-100 years' when (
            cond.stdage between 101 
            and 998
          ) then '100+ years' else 'w/o age - row not selectable' end, 
          cond.stdorgcd, 
          cond.statecd, 
          cond.unitcd, 
          cond.trtcd1, 
          cond.trtcd2, 
          cond.trtcd3, 
          refftgrp.meaning, 
          cond.stdage, 
          cond.dstrbcd1, 
          cond.dstrbcd2, 
          cond.dstrbcd3 
          /*, refft.typgrpcd */
          ) 
    GROUP BY 
      conusRegion, 
      cube(
        possibleTypes, standOrigin, ageBin
      ) 
    ORDER BY 
      conusRegion, 
      possibleTypes, 
      standOrigin, 
      ageBin
  ) num 
  join (
    SELECT 
      conusRegion, 
      coalesce(possibleTypes, 'Unknown') as possibleTypes, 
      coalesce(standOrigin, 'Unknown') as standOrigin, 
      coalesce(ageBin, 'Unknown') as ageBin, 
      sum(ESTIMATED_VALUE * EXPNS) ESTIMATE 
    FROM 
      (
        SELECT 
          case when (
            cond.stdage between 0 
            and 20
          ) then '0-20 years' when (
            cond.stdage between 21 
            and 40
          ) then '21-40 years' when (
            cond.stdage between 41 
            and 60
          ) then '41-60 years' when (
            cond.stdage between 61 
            and 80
          ) then '61-80 years' when (
            cond.stdage between 81 
            and 100
          ) then '81-100 years' when (
            cond.stdage between 101 
            and 998
          ) then '100+ years' else 'w/o age - row not selectable' end as agebin, 
          refftgrp.meaning as possibleTypes, 
          case when cond.stdorgcd = 1 then 'Planted' else 'Natural' end as standOrigin, 
          case when cond.statecd in(
            9, 10, 11, 23, 24, 25, 33, 34, 36, 42, 44, 
            50, 54
          ) then 'Northeast' when cond.statecd in(17, 18, 19, 29, 39) then 'Central States' when cond.statecd in(20, 31, 38, 46) then 'Great Plains' when cond.statecd in(26, 27, 55) then 'Northern Lake States' when (
            cond.statecd in(40, 48) 
            and cond.unitcd in(3, 4, 5, 6, 7)
          ) then 'Great Plains' when cond.statecd in(1, 5, 21, 22, 28, 40, 47, 48) then 'South Central' when cond.statecd in(12, 13, 37, 45, 51) then 'Southeast' when cond.statecd in(16, 30) then 'Rocky Mountain North' when cond.statecd in(4, 8, 32, 35, 49, 56) then 'Rocky Mountain South' when cond.statecd in(6) then 'Pacific Southwest' when (
            cond.statecd in(41, 53) 
            and cond.unitcd in(0, 1, 2, 5, 6, 7)
          ) then 'Pacific Northwest Westside' when (
            cond.statecd in(41, 53) 
            and cond.unitcd in(3, 4, 8, 9)
          ) then 'Pacific Northwest Eastside' else 'not located' end as conusRegion, 
          pop_stratum.expns EXPNS, 
          SUM(
            SUBPTYP_PROP_CHNG *.25 * CASE COND.PROP_BASIS WHEN 'MACR' THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE POP_STRATUM.ADJ_FACTOR_SUBP END
          ) AS ESTIMATED_VALUE 
        FROM 
          FS_FIADB.COND PCOND, 
          FS_FIADB.SUBP_COND_CHNG_MTRX, 
          FS_FIADB.POP_STRATUM POP_STRATUM 
          JOIN FS_FIADB.POP_PLOT_STRATUM_ASSGN ON (
            POP_PLOT_STRATUM_ASSGN.STRATUM_CN = POP_STRATUM.CN
          ) 
          JOIN FS_FIADB.PLOT ON (
            POP_PLOT_STRATUM_ASSGN.PLT_CN = PLOT.CN
          ) 
          JOIN FS_FIADB.PLOTGEOM ON (PLOT.CN = PLOTGEOM.CN) 
          JOIN FS_FIADB.COND ON (COND.PLT_CN = PLOT.CN) 
          JOIN FS_FIADB.REF_FOREST_TYPE refft ON (COND.FORTYPCD = refft.VALUE) 
          JOIN FS_FIADB.REF_FOREST_TYPE_GROUP refftgrp ON (refftgrp.value = refft.typgrpcd) 
        WHERE 
          SUBP_COND_CHNG_MTRX.CONDID = COND.CONDID 
          AND SUBP_COND_CHNG_MTRX.PLT_CN = COND.PLT_CN 
          AND PCOND.CONDID = SUBP_COND_CHNG_MTRX.PREVCOND 
          AND PCOND.PLT_CN = SUBP_COND_CHNG_MTRX.PREV_PLT_CN 
          AND PCOND.cond_status_cd = 1 
          and SUBP_COND_CHNG_MTRX.SUBPTYP = 1 
          AND COND.COND_STATUS_CD = 1 
          AND COND.CONDPROP_UNADJ IS NOT NULL 
         /* conus46 and the 2 split states */

         AND (
            ( pop_stratum.evalid in (12203, 41903, 52103, 61903, 81903, 92103, 
             102103, 121903, 132103, 161903, 172103, 182103, 192103, 202003, 211903, 221903, 
             232103, 242103, 252003, 262103, 272103, 282103, 292103, 301903, 312103, 321903,
             332103, 342003, 351903, 362003, 372203, 382103, 392103, 411903, 422103, 442103, 
             452103, 462103, 471903, 491903, 502103, 512103, 531903, 542103, 552103, 561803	))  -- the 46  
             or
            (POP_PLOT_STRATUM_ASSGN.unitcd in (3,4,5,6,7) and pop_stratum.evalid = 481903)  -- TX west  
            or
            (POP_PLOT_STRATUM_ASSGN.unitcd in (1,2) and pop_stratum.evalid = 482123)  -- TX east 
            or 
            (POP_PLOT_STRATUM_ASSGN.unitcd in (3,4,5,6,7) and pop_stratum.evalid = 402003)  -- OK west   
            or
            (POP_PLOT_STRATUM_ASSGN.unitcd in (1,2) and pop_stratum.evalid = 402003)  -- OK east  
          )
			
        GROUP BY 
          pop_stratum.estn_unit_cn, 
          pop_stratum.cn, 
          plot.cn, 
          plot.lat, 
          plot.lon, 
          pop_stratum.expns, 
          FS_FIADB.EVALIDATORFUNCTIONS.fortypgrpcdLabel(cond.fortypcd), 
          case when (
            cond.stdage between 0 
            and 20
          ) then '0-20 years' when (
            cond.stdage between 21 
            and 40
          ) then '21-40 years' when (
            cond.stdage between 41 
            and 60
          ) then '41-60 years' when (
            cond.stdage between 61 
            and 80
          ) then '61-80 years' when (
            cond.stdage between 81 
            and 100
          ) then '81-100 years' when (
            cond.stdage between 101 
            and 998
          ) then '100+ years' else 'w/o age - row not selectable' end, 
          cond.stdorgcd, 
          cond.statecd, 
          cond.unitcd, 
          cond.trtcd1, 
          cond.trtcd2, 
          cond.trtcd3, 
          refftgrp.meaning, 
          cond.stdage, 
          cond.dstrbcd1, 
          cond.dstrbcd2, 
          cond.dstrbcd3 
          /*, refft.typgrpcd*/
          ) 
    GROUP BY 
      conusRegion, 
      cube(
        possibleTypes, standOrigin, ageBin
      ) 
    ORDER BY 
      conusRegion, 
      possibleTypes, 
      standOrigin, 
      ageBin
  ) den ON num.conusRegion = den.conusRegion 
  AND num.possibleTypes = den.possibleTypes 
  AND num.standOrigin = den.standOrigin 
  AND num.ageBin = den.ageBin 
union 
SELECT 
  num.conusRegion, 
  num.possibleTypes, 
  num.standOrigin, 
  num.ageBin, 
  num.ESTIMATEag / den.ESTIMATE as grossGrowCag_tpapy, 
  /* RATIO_ESTIMATED_VALUE,  */
  num.ESTIMATEbg / den.ESTIMATE as grossGrowCbg_tpapy, 
  den.ESTIMATE as forest_acres, 
  num.t2_survivor_agdw_ton / den.ESTIMATE as t2_survivorCag_tpapy, 
  num.t2_ingrowth_agdw_ton / den.ESTIMATE as t2_ingrowthCag_tpapy, 
  num.t2_reversion_agdw_ton / den.ESTIMATE as t2_reversionCag_tpapy, 
  num.t2_cut_agdw_ton / den.ESTIMATE as t2_cutCag_tpapy, 
  num.t2_diversion_agdw_ton / den.ESTIMATE as t2_diversionCag_tpapy, 
  num.t2_mortality_agdw_ton / den.ESTIMATE as t2_mortalityCag_tpapy, 
  num.t1_survivor_agdw_ton / den.ESTIMATE as t1_survivorCag_tpapy, 
  num.t1_cut_agdw_ton / den.ESTIMATE as t1_cutCag_tpapy, 
  num.t1_diversion_agdw_ton / den.ESTIMATE as t1_diversionCag_tpapy, 
  num.t1_mortality_agdw_ton / den.ESTIMATE as t1_mortalityCag_tpapy, 
  num.t2_survivor_bgdw_ton / den.ESTIMATE as t2_survivorCbg_tpapy, 
  num.t2_ingrowth_bgdw_ton / den.ESTIMATE as t2_ingrowthCbg_tpapy, 
  num.t2_reversion_bgdw_ton / den.ESTIMATE as t2_reversionCbg_tpapy, 
  num.t2_cut_bgdw_ton / den.ESTIMATE as t2_cutCbg_tpapy, 
  num.t2_diversion_bgdw_ton / den.ESTIMATE as t2_diversionCbg_tpapy, 
  num.t2_mortality_bgdw_ton / den.ESTIMATE as t2_mortalityCbg_tpapy, 
  num.t1_survivor_bgdw_ton / den.ESTIMATE as t1_survivorCbg_tpapy, 
  num.t1_cut_bgdw_ton / den.ESTIMATE as t1_cutCbg_tpapy, 
  num.t1_diversion_bgdw_ton / den.ESTIMATE as t1_diversionCbg_tpapy, 
  num.t1_mortality_bgdw_ton / den.ESTIMATE as t1_mortalityCbg_tpapy 
FROM 
  (
    SELECT 
      conusRegion, 
      coalesce(possibleTypes, 'Unknown') as possibleTypes, 
      coalesce(standOrigin, 'Unknown') as standOrigin, 
      coalesce(ageBin, 'Unknown') as ageBin, 
      sum(ESTIMATED_VALUEag * EXPNS) ESTIMATEag, 
      sum(ESTIMATED_VALUEbg * EXPNS) ESTIMATEbg, 
      sum(t2_survivor_agdw_ton * EXPNS) t2_survivor_agdw_ton, 
      sum(t2_ingrowth_agdw_ton * EXPNS) t2_ingrowth_agdw_ton, 
      sum(t2_reversion_agdw_ton * EXPNS) t2_reversion_agdw_ton, 
      sum(t2_cut_agdw_ton * EXPNS) t2_cut_agdw_ton, 
      sum(t2_diversion_agdw_ton * EXPNS) t2_diversion_agdw_ton, 
      sum(t2_mortality_agdw_ton * EXPNS) t2_mortality_agdw_ton, 
      sum(t1_survivor_agdw_ton * EXPNS) t1_survivor_agdw_ton, 
      sum(t1_cut_agdw_ton * EXPNS) t1_cut_agdw_ton, 
      sum(t1_diversion_agdw_ton * EXPNS) t1_diversion_agdw_ton, 
      sum(t1_mortality_agdw_ton * EXPNS) t1_mortality_agdw_ton, 
      sum(t2_survivor_bgdw_ton * EXPNS) t2_survivor_bgdw_ton, 
      sum(t2_ingrowth_bgdw_ton * EXPNS) t2_ingrowth_bgdw_ton, 
      sum(t2_reversion_bgdw_ton * EXPNS) t2_reversion_bgdw_ton, 
      sum(t2_cut_bgdw_ton * EXPNS) t2_cut_bgdw_ton, 
      sum(t2_diversion_bgdw_ton * EXPNS) t2_diversion_bgdw_ton, 
      sum(t2_mortality_bgdw_ton * EXPNS) t2_mortality_bgdw_ton, 
      sum(t1_survivor_bgdw_ton * EXPNS) t1_survivor_bgdw_ton, 
      sum(t1_cut_bgdw_ton * EXPNS) t1_cut_bgdw_ton, 
      sum(t1_diversion_bgdw_ton * EXPNS) t1_diversion_bgdw_ton, 
      sum(t1_mortality_bgdw_ton * EXPNS) t1_mortality_bgdw_ton 
    FROM 
      (
        SELECT 
          case when (
            cond.stdage between 0 
            and 20
          ) then '0-20 years' when (
            cond.stdage between 21 
            and 40
          ) then '21-40 years' when (
            cond.stdage between 41 
            and 60
          ) then '41-60 years' when (
            cond.stdage between 61 
            and 80
          ) then '61-80 years' when (
            cond.stdage between 81 
            and 100
          ) then '81-100 years' when (
            cond.stdage between 101 
            and 998
          ) then '100+ years' else 'w/o age - row not selectable' end as agebin, 
          case when refft.typgrpcd in (180, 970) then 'predominantly woodland species, type not known' when refft.typgrpcd in (999) then 'not sure, very few trees (ie, nonstocked)' when refft.typgrpcd < 400 then 'predominantly softwood species, type not known' else 'predominantly hardwood species, type not known' end as possibleTypes, 
          case when cond.stdorgcd = 1 then 'Planted' else 'Natural' end as standOrigin, 
          case when cond.statecd in(
            9, 10, 11, 23, 24, 25, 33, 34, 36, 42, 44, 
            50, 54
          ) then 'Northeast' when cond.statecd in(17, 18, 19, 29, 39) then 'Central States' when cond.statecd in(20, 31, 38, 46) then 'Great Plains' when cond.statecd in(26, 27, 55) then 'Northern Lake States' when (
            cond.statecd in(40, 48) 
            and cond.unitcd in(3, 4, 5, 6, 7)
          ) then 'Great Plains' when cond.statecd in(1, 5, 21, 22, 28, 40, 47, 48) then 'South Central' when cond.statecd in(12, 13, 37, 45, 51) then 'Southeast' when cond.statecd in(16, 30) then 'Rocky Mountain North' when cond.statecd in(4, 8, 32, 35, 49, 56) then 'Rocky Mountain South' when cond.statecd in(6) then 'Pacific Southwest' when (
            cond.statecd in(41, 53) 
            and cond.unitcd in(0, 1, 2, 5, 6, 7)
          ) then 'Pacific Northwest Westside' when (
            cond.statecd in(41, 53) 
            and cond.unitcd in(3, 4, 8, 9)
          ) then 'Pacific Northwest Eastside' else 'not located' end as conusRegion, 
          pop_stratum.expns EXPNS, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * (
              CASE WHEN BE.ONEORTWO = 2 THEN (
                CASE WHEN (
                  GRM.COMPONENT = 'SURVIVOR' 
                  OR GRM.COMPONENT = 'INGROWTH' 
                  OR GRM.COMPONENT LIKE 'REVERSION%'
                ) THEN (
                  TREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
                ) WHEN (
                  GRM.COMPONENT LIKE 'CUT%' 
                  OR GRM.COMPONENT LIKE 'DIVERSION%' 
                  OR GRM.COMPONENT LIKE 'MORTALITY%'
                ) THEN (
                  TRE_MIDPT.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
                ) ELSE (0) END
              ) ELSE (
                CASE WHEN (
                  GRM.COMPONENT = 'SURVIVOR' 
                  OR GRM.COMPONENT = 'CUT1' 
                  OR GRM.COMPONENT = 'DIVERSION1' 
                  OR GRM.COMPONENT = 'MORTALITY1'
                ) THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
                  TRE_BEGIN.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
                ) ELSE -(
                  PTREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
                ) END ELSE (0) END
              ) END
            )
          ) AS ESTIMATED_VALUEag, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * (
              CASE WHEN BE.ONEORTWO = 2 THEN (
                CASE WHEN (
                  GRM.COMPONENT = 'SURVIVOR' 
                  OR GRM.COMPONENT = 'INGROWTH' 
                  OR GRM.COMPONENT LIKE 'REVERSION%'
                ) THEN (
                  TREE.DRYBIO_bG / 2 / 2000 / PLOT.REMPER
                ) WHEN (
                  GRM.COMPONENT LIKE 'CUT%' 
                  OR GRM.COMPONENT LIKE 'DIVERSION%' 
                  OR GRM.COMPONENT LIKE 'MORTALITY%'
                ) THEN (
                  TRE_MIDPT.DRYBIO_bG / 2 / 2000 / PLOT.REMPER
                ) ELSE (0) END
              ) ELSE (
                CASE WHEN (
                  GRM.COMPONENT = 'SURVIVOR' 
                  OR GRM.COMPONENT = 'CUT1' 
                  OR GRM.COMPONENT = 'DIVERSION1' 
                  OR GRM.COMPONENT = 'MORTALITY1'
                ) THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
                  TRE_BEGIN.DRYBIO_bG / 2 / 2000 / PLOT.REMPER
                ) ELSE -(
                  PTREE.DRYBIO_bG / 2 / 2000 / PLOT.REMPER
                ) END ELSE (0) END
              ) END
            )
          ) AS ESTIMATED_VALUEbg, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT = 'SURVIVOR' THEN TREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_survivor_agdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT = 'INGROWTH' THEN TREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_ingrowth_agdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'REVERSION%' THEN TREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_reversion_agdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'CUT%' THEN TRE_MIDPT.DRYBIO_AG / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_cut_agdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'DIVERSION%' THEN TRE_MIDPT.DRYBIO_AG / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_diversion_agdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'MORTALITY%' THEN TRE_MIDPT.DRYBIO_AG / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_mortality_agdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'SURVIVOR' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_survivor_agdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'CUT1' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_cut_agdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'DIVERSION1' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_diversion_agdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'MORTALITY1' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_AG / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_mortality_agdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT = 'SURVIVOR' THEN TREE.DRYBIO_bg / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_survivor_bgdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT = 'INGROWTH' THEN TREE.DRYBIO_bg / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_ingrowth_bgdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'REVERSION%' THEN TREE.DRYBIO_bg / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_reversion_bgdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'CUT%' THEN TRE_MIDPT.DRYBIO_bg / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_cut_bgdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'DIVERSION%' THEN TRE_MIDPT.DRYBIO_bg / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_diversion_bgdw_ton, 
          SUM(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 2 THEN CASE WHEN GRM.COMPONENT LIKE 'MORTALITY%' THEN TRE_MIDPT.DRYBIO_bg / 2 / 2000 / PLOT.REMPER else (0) end end
          ) as t2_mortality_bgdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'SURVIVOR' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_survivor_bgdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'CUT1' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_cut_bgdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'DIVERSION1' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_diversion_bgdw_ton, 
          sum(
            GRM.TPAGROW_UNADJ * (
              CASE WHEN COALESCE(GRM.SUBPTYP_GRM, 0) = 0 THEN (0) WHEN GRM.SUBPTYP_GRM = 1 THEN POP_STRATUM.ADJ_FACTOR_SUBP WHEN GRM.SUBPTYP_GRM = 2 THEN POP_STRATUM.ADJ_FACTOR_MICR WHEN GRM.SUBPTYP_GRM = 3 THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE (0) END
            ) * CASE WHEN BE.ONEORTWO = 1 THEN CASE WHEN GRM.COMPONENT = 'MORTALITY1' THEN CASE WHEN TRE_BEGIN.TRE_CN IS NOT NULL THEN -(
              TRE_BEGIN.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) ELSE -(
              PTREE.DRYBIO_bg / 2 / 2000 / PLOT.REMPER
            ) end ELSE (0) end END
          ) as t1_mortality_bgdw_ton 
        FROM 
          FS_FIADB.BEGINEND BE, 
          FS_FIADB.POP_STRATUM POP_STRATUM 
          JOIN FS_FIADB.POP_PLOT_STRATUM_ASSGN POP_PLOT_STRATUM_ASSGN ON (
            POP_STRATUM.CN = POP_PLOT_STRATUM_ASSGN.STRATUM_CN
          ) 
          JOIN FS_FIADB.PLOT PLOT ON (
            POP_PLOT_STRATUM_ASSGN.PLT_CN = PLOT.CN
          ) 
          JOIN FS_FIADB.PLOTGEOM PLOTGEOM ON (PLOT.CN = PLOTGEOM.CN) 
          JOIN FS_FIADB.PLOT PPLOT ON (PLOT.PREV_PLT_CN = PPLOT.CN) 
          JOIN FS_FIADB.COND PCOND ON (PLOT.PREV_PLT_CN = PCOND.PLT_CN) 
          JOIN FS_FIADB.COND COND ON (PLOT.CN = COND.PLT_CN) 
          JOIN FS_FIADB.REF_FOREST_TYPE refft ON (COND.FORTYPCD = refft.VALUE) 
          JOIN FS_FIADB.REF_FOREST_TYPE_GROUP refftgrp ON (refftgrp.value = refft.typgrpcd) 
          JOIN FS_FIADB.TREE TREE ON (
            TREE.CONDID = COND.CONDID 
            AND TREE.PLT_CN = PLOT.CN 
            AND TREE.PREVCOND = PCOND.CONDID
          ) 
          LEFT OUTER JOIN FS_FIADB.TREE PTREE ON (TREE.PREV_TRE_CN = PTREE.CN) 
          LEFT OUTER JOIN FS_FIADB.TREE_GRM_BEGIN TRE_BEGIN ON (TREE.CN = TRE_BEGIN.TRE_CN) 
          LEFT OUTER JOIN FS_FIADB.TREE_GRM_MIDPT TRE_MIDPT ON (TREE.CN = TRE_MIDPT.TRE_CN) 
          LEFT OUTER JOIN (
            SELECT 
              TRE_CN, 
              DIA_BEGIN, 
              DIA_MIDPT, 
              DIA_END, 
              MICR_COMPONENT_AL_FOREST AS COMPONENT, 
              MICR_SUBPTYP_GRM_AL_FOREST AS SUBPTYP_GRM, 
              MICR_TPAGROW_UNADJ_AL_FOREST AS TPAGROW_UNADJ 
            FROM 
              FS_FIADB.TREE_GRM_COMPONENT
          ) GRM ON (TREE.CN = GRM.TRE_CN) 
        WHERE 
          TREE.PREVCOND = PCOND.CONDID 
          AND PCOND.COND_STATUS_CD = 1 
          AND COND.COND_STATUS_CD = 1 
          AND 1 = 1 
          /* conus46 and the 2 split states */

         AND (
            ( pop_stratum.evalid in (12203, 41903, 52103, 61903, 81903, 92103, 
             102103, 121903, 132103, 161903, 172103, 182103, 192103, 202003, 211903, 221903, 
             232103, 242103, 252003, 262103, 272103, 282103, 292103, 301903, 312103, 321903,
             332103, 342003, 351903, 362003, 372203, 382103, 392103, 411903, 422103, 442103, 
             452103, 462103, 471903, 491903, 502103, 512103, 531903, 542103, 552103, 561803	))  -- the 46  
             or
            (POP_PLOT_STRATUM_ASSGN.unitcd in (3,4,5,6,7) and pop_stratum.evalid = 481903)  -- TX west  
            or
            (POP_PLOT_STRATUM_ASSGN.unitcd in (1,2) and pop_stratum.evalid = 482123)  -- TX east 
            or 
            (POP_PLOT_STRATUM_ASSGN.unitcd in (3,4,5,6,7) and pop_stratum.evalid = 402003)  -- OK west   
            or
            (POP_PLOT_STRATUM_ASSGN.unitcd in (1,2) and pop_stratum.evalid = 402003)  -- OK east  
          )
			
        GROUP BY 
          pop_stratum.estn_unit_cn, 
          pop_stratum.cn, 
          plot.cn, 
          plot.lat, 
          plot.lon, 
          pop_stratum.expns, 
          FS_FIADB.EVALIDATORFUNCTIONS.fortypgrpcdLabel(cond.fortypcd), 
          case when (
            cond.stdage between 0 
            and 20
          ) then '0-20 years' when (
            cond.stdage between 21 
            and 40
          ) then '21-40 years' when (
            cond.stdage between 41 
            and 60
          ) then '41-60 years' when (
            cond.stdage between 61 
            and 80
          ) then '61-80 years' when (
            cond.stdage between 81 
            and 100
          ) then '81-100 years' when (
            cond.stdage between 101 
            and 998
          ) then '100+ years' else 'w/o age - row not selectable' end, 
          cond.stdorgcd, 
          cond.statecd, 
          cond.unitcd, 
          cond.trtcd1, 
          cond.trtcd2, 
          cond.trtcd3, 
          refftgrp.meaning, 
          cond.stdage, 
          cond.dstrbcd1, 
          cond.dstrbcd2, 
          cond.dstrbcd3, 
          refft.typgrpcd
      ) 
    GROUP BY 
      conusRegion, 
      cube(
        possibleTypes, standOrigin, ageBin
      ) 
    ORDER BY 
      conusRegion, 
      possibleTypes, 
      standOrigin, 
      ageBin
  ) num 
  join (
    SELECT 
      conusRegion, 
      coalesce(possibleTypes, 'Unknown') as possibleTypes, 
      coalesce(standOrigin, 'Unknown') as standOrigin, 
      coalesce(ageBin, 'Unknown') as ageBin, 
      sum(ESTIMATED_VALUE * EXPNS) ESTIMATE 
    FROM 
      (
        SELECT 
          case when (
            cond.stdage between 0 
            and 20
          ) then '0-20 years' when (
            cond.stdage between 21 
            and 40
          ) then '21-40 years' when (
            cond.stdage between 41 
            and 60
          ) then '41-60 years' when (
            cond.stdage between 61 
            and 80
          ) then '61-80 years' when (
            cond.stdage between 81 
            and 100
          ) then '81-100 years' when (
            cond.stdage between 101 
            and 998
          ) then '100+ years' else 'w/o age - row not selectable' end as agebin, 
          case when refft.typgrpcd in (180, 970) then 'predominantly woodland species, type not known' when refft.typgrpcd in (999) then 'not sure, very few trees (ie, nonstocked)' when refft.typgrpcd < 400 then 'predominantly softwood species, type not known' else 'predominantly hardwood species, type not known' end as possibleTypes, 
          case when cond.stdorgcd = 1 then 'Planted' else 'Natural' end as standOrigin, 
          
          /*to_char(cond.statecd)  statecd, */
          case when cond.statecd in(
            9, 10, 11, 23, 24, 25, 33, 34, 36, 42, 44, 
            50, 54
          ) then 'Northeast' when cond.statecd in(17, 18, 19, 29, 39) then 'Central States' when cond.statecd in(20, 31, 38, 46) then 'Great Plains' when cond.statecd in(26, 27, 55) then 'Northern Lake States' when (
            cond.statecd in(40, 48) 
            and cond.unitcd in(3, 4, 5, 6, 7)
          ) then 'Great Plains' when cond.statecd in(1, 5, 21, 22, 28, 40, 47, 48) then 'South Central' when cond.statecd in(12, 13, 37, 45, 51) then 'Southeast' when cond.statecd in(16, 30) then 'Rocky Mountain North' when cond.statecd in(4, 8, 32, 35, 49, 56) then 'Rocky Mountain South' when cond.statecd in(6) then 'Pacific Southwest' when (
            cond.statecd in(41, 53) 
            and cond.unitcd in(0, 1, 2, 5, 6, 7)
          ) then 'Pacific Northwest Westside' when (
            cond.statecd in(41, 53) 
            and cond.unitcd in(3, 4, 8, 9)
          ) then 'Pacific Northwest Eastside' else 'not located' end as conusRegion, 
          pop_stratum.expns EXPNS, 
          SUM(
            SUBPTYP_PROP_CHNG *.25 * CASE COND.PROP_BASIS WHEN 'MACR' THEN POP_STRATUM.ADJ_FACTOR_MACR ELSE POP_STRATUM.ADJ_FACTOR_SUBP END
          ) AS ESTIMATED_VALUE 
        FROM 
          FS_FIADB.COND PCOND, 
          FS_FIADB.SUBP_COND_CHNG_MTRX, 
          FS_FIADB.POP_STRATUM POP_STRATUM 
          JOIN FS_FIADB.POP_PLOT_STRATUM_ASSGN ON (
            POP_PLOT_STRATUM_ASSGN.STRATUM_CN = POP_STRATUM.CN
          ) 
          JOIN FS_FIADB.PLOT ON (
            POP_PLOT_STRATUM_ASSGN.PLT_CN = PLOT.CN
          ) 
          JOIN FS_FIADB.PLOTGEOM ON (PLOT.CN = PLOTGEOM.CN) 
          JOIN FS_FIADB.COND ON (COND.PLT_CN = PLOT.CN) 
          JOIN FS_FIADB.REF_FOREST_TYPE refft ON (COND.FORTYPCD = refft.VALUE) 
          JOIN FS_FIADB.REF_FOREST_TYPE_GROUP refftgrp ON (refftgrp.value = refft.typgrpcd) 
        WHERE 
          SUBP_COND_CHNG_MTRX.CONDID = COND.CONDID 
          AND SUBP_COND_CHNG_MTRX.PLT_CN = COND.PLT_CN 
          AND PCOND.CONDID = SUBP_COND_CHNG_MTRX.PREVCOND 
          AND PCOND.PLT_CN = SUBP_COND_CHNG_MTRX.PREV_PLT_CN 
          AND PCOND.cond_status_cd = 1 
          and SUBP_COND_CHNG_MTRX.SUBPTYP = 1 
          AND COND.COND_STATUS_CD = 1 
          AND COND.CONDPROP_UNADJ IS NOT NULL 
         /* conus46 and the 2 split states */

         AND (
            ( pop_stratum.evalid in (12203, 41903, 52103, 61903, 81903, 92103, 
             102103, 121903, 132103, 161903, 172103, 182103, 192103, 202003, 211903, 221903, 
             232103, 242103, 252003, 262103, 272103, 282103, 292103, 301903, 312103, 321903,
             332103, 342003, 351903, 362003, 372203, 382103, 392103, 411903, 422103, 442103, 
             452103, 462103, 471903, 491903, 502103, 512103, 531903, 542103, 552103, 561803	))  -- the 46  
             or
            (POP_PLOT_STRATUM_ASSGN.unitcd in (3,4,5,6,7) and pop_stratum.evalid = 481903)  -- TX west  
            or
            (POP_PLOT_STRATUM_ASSGN.unitcd in (1,2) and pop_stratum.evalid = 482123)  -- TX east 
            or 
            (POP_PLOT_STRATUM_ASSGN.unitcd in (3,4,5,6,7) and pop_stratum.evalid = 402003)  -- OK west   
            or
            (POP_PLOT_STRATUM_ASSGN.unitcd in (1,2) and pop_stratum.evalid = 402003)  -- OK east  
          )
			
        GROUP BY 
          pop_stratum.estn_unit_cn, 
          pop_stratum.cn, 
          plot.cn, 
          plot.lat, 
          plot.lon, 
          pop_stratum.expns, 
          FS_FIADB.EVALIDATORFUNCTIONS.fortypgrpcdLabel(cond.fortypcd), 
          case when (
            cond.stdage between 0 
            and 20
          ) then '0-20 years' when (
            cond.stdage between 21 
            and 40
          ) then '21-40 years' when (
            cond.stdage between 41 
            and 60
          ) then '41-60 years' when (
            cond.stdage between 61 
            and 80
          ) then '61-80 years' when (
            cond.stdage between 81 
            and 100
          ) then '81-100 years' when (
            cond.stdage between 101 
            and 998
          ) then '100+ years' else 'w/o age - row not selectable' end, 
          cond.stdorgcd, 
          cond.statecd, 
          cond.unitcd, 
          cond.trtcd1, 
          cond.trtcd2, 
          cond.trtcd3, 
          refftgrp.meaning, 
          cond.stdage, 
          cond.dstrbcd1, 
          cond.dstrbcd2, 
          cond.dstrbcd3, 
          refft.typgrpcd
      ) 
    GROUP BY 
      conusRegion, 
      cube(
        possibleTypes, standOrigin, ageBin
      ) 
    ORDER BY 
      conusRegion, 
      possibleTypes, 
      standOrigin, 
      ageBin
  ) den ON num.conusRegion = den.conusRegion 
  AND num.possibleTypes = den.possibleTypes 
  AND num.standOrigin = den.standOrigin 
  AND num.ageBin = den.ageBin 
ORDER BY 
  conusRegion, 
  possibleTypes, 
  standOrigin, 
  ageBin
